jQuery(document).ready(function($) {
    'use strict';
    
    var selectedFiles = [];
    var currentDirectory = '';
    var directoryHistory = [];
    
    // Browse FTP server
    $('#scan-ftp-files').on('click', function(e) {
        e.preventDefault();
        browseDirectory('');
    });
    
    // Go back button
    $('#go-back').on('click', function(e) {
        e.preventDefault();
        if (directoryHistory.length > 0) {
            var previousDir = directoryHistory.pop();
            browseDirectory(previousDir);
        }
    });
    
    // Browse directory function
    function browseDirectory(directory) {
        var $button = $('#scan-ftp-files');
        var $results = $('#browser-results');
        var $contents = $('#browser-contents');
        var $currentPath = $('#current-path');
        var $goBack = $('#go-back');
        
        // Disable button and show loading
        $button.prop('disabled', true).text('Loading...');
        $results.hide();
        
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'scan_ftp_files',
                nonce: ftp_media_admin.nonce,
                directory: directory
            },
            timeout: 300000, // 5 minutes timeout for scanning
            success: function(response) {
                if (response.success) {
                    currentDirectory = directory;
                    displayDirectoryContents(response.data.contents, directory);
                    $results.show();
                    $currentPath.text('Current: ' + (directory || '/'));
                    $goBack.toggle(directoryHistory.length > 0);
                    $button.prop('disabled', false).text('Browse FTP Server');
                } else {
                    alert('Error: ' + response.data);
                    $button.prop('disabled', false).text('Browse FTP Server');
                }
            },
            error: function(xhr, status, error) {
                var errorMessage = 'Failed to browse FTP server';
                
                if (status === 'timeout') {
                    errorMessage = 'Browse timed out. Please check your FTP server settings and try again.';
                } else if (xhr.responseJSON && xhr.responseJSON.data) {
                    errorMessage = xhr.responseJSON.data;
                } else if (error) {
                    errorMessage = 'Network error: ' + error;
                }
                
                alert('Error: ' + errorMessage);
                $button.prop('disabled', false).text('Browse FTP Server');
            }
        });
    }
    
    // Display directory contents
    function displayDirectoryContents(contents, directory) {
        var html = '<div class="ftp-browser-contents">';
        html += '<p><strong>Found ' + contents.length + ' items in directory:</strong></p>';
        html += '<div class="ftp-contents-grid">';
        
        // Sort contents: directories first, then files
        contents.sort(function(a, b) {
            if (a.type === 'directory' && b.type === 'file') return -1;
            if (a.type === 'file' && b.type === 'directory') return 1;
            return a.name.localeCompare(b.name);
        });
        
        contents.forEach(function(item, index) {
            var itemSize = item.type === 'file' ? formatFileSize(item.size) : '';
            var itemDate = formatDate(item.date);
            var isDirectory = item.type === 'directory';
            
            html += '<div class="ftp-item" data-path="' + item.path + '" data-type="' + item.type + '">';
            
            // Checkbox for selection
            html += '<div class="item-checkbox-container">';
            html += '<input type="checkbox" class="item-checkbox" data-file=\'' + JSON.stringify(item) + '\'>';
            html += '</div>';
            
            // Main content area
            html += '<div class="item-content">';
            html += '<div class="item-icon">' + item.icon + '</div>';
            html += '<div class="item-info">';
            html += '<div class="item-name">' + item.name + '</div>';
            html += '<div class="item-meta">';
            
            if (isDirectory) {
                html += '<div class="meta-item">';
                html += '<span class="meta-icon">📁</span>';
                html += '<span class="meta-label">Folder</span>';
                html += '</div>';
            } else {
                html += '<div class="meta-item">';
                html += '<span class="meta-icon">📄</span>';
                html += '<span class="meta-label">File</span>';
                html += '</div>';
                
                if (itemSize) {
                    html += '<div class="meta-item">';
                    html += '<span class="meta-icon">💾</span>';
                    html += '<span class="meta-label">' + itemSize + '</span>';
                    html += '</div>';
                }
            }
            
            html += '<div class="meta-item">';
            html += '<span class="meta-icon">📅</span>';
            html += '<span class="meta-label">' + itemDate + '</span>';
            html += '</div>';
            
            html += '</div>';
            html += '</div>';
            html += '</div>';
            
            // Navigation button for folders
            if (isDirectory) {
                html += '<div class="item-actions">';
                html += '<button type="button" class="button button-small navigate-btn" data-path="' + item.path + '">Open</button>';
                html += '</div>';
            }
            
            html += '</div>';
        });
        
        html += '</div></div>';
        
        $('#browser-contents').html(html);
        
        // Show import button if any items found
        if (contents.length > 0) {
            $('#import-selected-files').show();
        }
        
        // Bind checkbox events
        bindFileEvents();
    }
    
    // Navigate to directory
    function navigateToDirectory(directoryPath) {
        directoryHistory.push(currentDirectory);
        browseDirectory(directoryPath);
    }
    
    // Format date as yy/mm/dd
    function formatDate(timestamp) {
        var date = new Date(timestamp * 1000);
        var year = date.getFullYear().toString().slice(-2);
        var month = (date.getMonth() + 1).toString().padStart(2, '0');
        var day = date.getDate().toString().padStart(2, '0');
        return year + '/' + month + '/' + day;
    }
    
    // Bind file selection events
    function bindFileEvents() {
        // Remove existing event handlers
        $('#select-all-files').off('change');
        $('#select-none-files').off('click');
        $('.item-checkbox').off('change');
        $('.navigate-btn').off('click');
        
        // Select all checkbox
        $('#select-all-files').on('change', function() {
            var isChecked = $(this).is(':checked');
            $('.item-checkbox').prop('checked', isChecked);
            updateSelectedFiles();
        });
        
        // Select none button
        $('#select-none-files').on('click', function() {
            $('.item-checkbox').prop('checked', false);
            $('#select-all-files').prop('checked', false);
            updateSelectedFiles();
        });
        
        // Individual item checkboxes
        $('.item-checkbox').on('change', function() {
            updateSelectedFiles();
        });
        
    // Navigation button clicks
    $('.navigate-btn').on('click', function(e) {
        e.preventDefault();
        e.stopPropagation();
        var path = $(this).data('path');
        
        console.log('Navigate button clicked:', path);
        navigateToDirectory(path);
    });
    
    // Refresh stats button
    $('#refresh-stats').on('click', function() {
        location.reload();
    });
    }
    
    // Update selected files array
    function updateSelectedFiles() {
        selectedFiles = [];
        $('.item-checkbox:checked').each(function() {
            var itemData = $(this).data('file');
            selectedFiles.push(itemData);
        });
        
        // Update select all checkbox state
        var totalItems = $('.item-checkbox').length;
        var checkedItems = $('.item-checkbox:checked').length;
        
        if (checkedItems === 0) {
            $('#select-all-files').prop('indeterminate', false).prop('checked', false);
        } else if (checkedItems === totalItems) {
            $('#select-all-files').prop('indeterminate', false).prop('checked', true);
        } else {
            $('#select-all-files').prop('indeterminate', true);
        }
        
        // Update import button
        if (selectedFiles.length > 0) {
            $('#import-selected-files').text('Import Selected Items (' + selectedFiles.length + ')').show();
        } else {
            $('#import-selected-files').hide();
        }
    }
    
    // Import selected files
    $('#import-selected-files').on('click', function(e) {
        e.preventDefault();
        
        if (selectedFiles.length === 0) {
            alert('Please select files to import.');
            return;
        }
        
        var $button = $(this);
        var $progress = $('#import-progress');
        var $status = $('#import-status');
        
        // Disable button and show progress
        $button.prop('disabled', true).text('Importing...');
        $progress.show();
        $status.html('<p>Starting import of ' + selectedFiles.length + ' files...</p>');
        
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'import_ftp_files',
                nonce: ftp_media_admin.nonce,
                files: selectedFiles
            },
            timeout: 300000, // 5 minutes timeout for import
                success: function(response) {
                    if (response.success) {
                        var result = response.data;
                        var message = '<div class="notice notice-success"><p><strong>Import Completed!</strong></p>';
                        message += '<ul>';
                        message += '<li>Imported: ' + result.imported + ' items</li>';
                        message += '<li>Skipped: ' + result.skipped + ' items (already exist)</li>';
                        message += '<li>Errors: ' + result.errors + ' items</li>';
                        message += '</ul>';
                        message += '<p><em>Note: Statistics will be updated when you refresh the page.</em></p></div>';

                        $status.html(message);
                        $button.prop('disabled', false).text('Import Selected Items');

                        // Don't auto-refresh, let user see the results
                        // User can manually refresh if they want to see updated stats
                    } else {
                        $status.html('<div class="notice notice-error"><p><strong>Import Failed:</strong> ' + response.data + '</p></div>');
                        $button.prop('disabled', false).text('Import Selected Items');
                    }
                },
            error: function(xhr, status, error) {
                var errorMessage = 'Failed to import files';
                
                if (status === 'timeout') {
                    errorMessage = 'Import timed out. Please try importing fewer files at once.';
                } else if (xhr.responseJSON && xhr.responseJSON.data) {
                    errorMessage = xhr.responseJSON.data;
                } else if (error) {
                    errorMessage = 'Network error: ' + error;
                }
                
                $status.html('<div class="notice notice-error"><p><strong>Import Failed:</strong> ' + errorMessage + '</p></div>');
                $button.prop('disabled', false).text('Import Selected Items');
            }
        });
    });
    
    // Format file size
    function formatFileSize(bytes) {
        if (bytes === 0) return '0 Bytes';
        
        var k = 1024;
        var sizes = ['Bytes', 'KB', 'MB', 'GB'];
        var i = Math.floor(Math.log(bytes) / Math.log(k));
        
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    }
});
