<?php
/**
 * Main FTP Media Plugin Class
 */

if (!defined('ABSPATH')) {
    exit;
}

class FTP_Media {
    
    private $settings;
    
    public function __construct() {
        add_action('init', array($this, 'init'));
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        
        // Add AJAX handlers
        add_action('wp_ajax_test_ftp_connection', array($this, 'ajax_test_ftp_connection'));
        add_action('wp_ajax_test_ftp_operations', array($this, 'ajax_test_ftp_operations'));
        add_action('wp_ajax_scan_ftp_files', array($this, 'ajax_scan_ftp_files'));
        add_action('wp_ajax_import_ftp_files', array($this, 'ajax_import_ftp_files'));
        add_action('wp_ajax_migrate_ftp_to_local', array($this, 'ajax_migrate_ftp_to_local'));
        
        // Initialize settings
        $this->settings = new FTP_Settings();
        
        // Hook into media upload process
        add_filter('wp_generate_attachment_metadata', array($this, 'handle_media_upload'), 10, 2);
        add_action('wp_handle_upload_prefilter', array($this, 'validate_upload'), 10, 1);
        
        // Hook into media delete process
        add_action('delete_attachment', array($this, 'handle_media_delete'), 10, 1);
        
        // Filter attachment URLs
        add_filter('wp_get_attachment_url', array($this, 'filter_attachment_url'), 10, 2);
        add_filter('wp_get_attachment_image_src', array($this, 'filter_attachment_image_src'), 10, 4);
    }
    
    public function init() {
        // Initialize plugin
        load_plugin_textdomain('ftp-media', false, dirname(plugin_basename(__FILE__)) . '/languages');
    }
    
    public function add_admin_menu() {
        // Main settings page
        add_options_page(
            __('FTP Media Settings', 'ftp-media'),
            __('FTP Media', 'ftp-media'),
            'manage_options',
            'ftp-media-settings',
            array($this, 'admin_page')
        );
        
        // Import page
        add_media_page(
            __('Import from FTP', 'ftp-media'),
            __('Import from FTP', 'ftp-media'),
            'manage_options',
            'ftp-media-import',
            array($this, 'import_page')
        );
    }
    
    public function enqueue_admin_scripts($hook) {
        // Load scripts on FTP Media pages
        if (strpos($hook, 'ftp-media') !== false || strpos($hook, 'media') !== false) {
            wp_enqueue_script('jquery');
            wp_enqueue_style('ftp-media-admin', FTP_MEDIA_PLUGIN_URL . 'assets/admin.css', array(), FTP_MEDIA_VERSION);
            wp_enqueue_script('ftp-media-admin', FTP_MEDIA_PLUGIN_URL . 'assets/admin.js', array('jquery'), FTP_MEDIA_VERSION, true);
            
            // Load import script on import page
            if (strpos($hook, 'ftp-media-import') !== false) {
                wp_enqueue_script('ftp-media-import', FTP_MEDIA_PLUGIN_URL . 'assets/import.js', array('jquery'), FTP_MEDIA_VERSION, true);
            }
            
            // Localize script for AJAX
            wp_localize_script('ftp-media-admin', 'ftp_media_admin', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('ftp_media_test_connection')
            ));
        }
    }
    
    public function admin_page() {
        $this->settings->render_settings_page();
    }
    
    public function import_page() {
        $importer = new FTP_Importer();
        $stats = $importer->get_import_stats();
        ?>
        <div class="wrap">
            <h1><?php _e('Import from FTP Server', 'ftp-media'); ?></h1>
            
                <div class="ftp-import-stats">
                    <h2><?php _e('Current Statistics', 'ftp-media'); ?></h2>
                    <button type="button" id="refresh-stats" class="button" style="float: right; margin-bottom: 10px;">
                        <?php _e('Refresh Stats', 'ftp-media'); ?>
                    </button>
                    <table class="widefat">
                        <tr>
                            <td><strong><?php _e('Total Attachments:', 'ftp-media'); ?></strong></td>
                            <td><?php echo $stats['total_attachments']; ?></td>
                        </tr>
                        <tr>
                            <td><strong><?php _e('FTP Attachments:', 'ftp-media'); ?></strong></td>
                            <td><?php echo $stats['ftp_attachments']; ?></td>
                        </tr>
                        <tr>
                            <td><strong><?php _e('Local Attachments:', 'ftp-media'); ?></strong></td>
                            <td><?php echo $stats['local_attachments']; ?></td>
                        </tr>
                    </table>
                </div>
            
            <div class="ftp-import-actions">
                <h2><?php _e('Import Actions', 'ftp-media'); ?></h2>
                <p><?php _e('Browse your FTP server for files that were manually uploaded and import them into WordPress.', 'ftp-media'); ?></p>
                
                <div class="ftp-browser">
                    <div class="ftp-browser-toolbar">
                        <button type="button" id="scan-ftp-files" class="button button-primary">
                            <?php _e('Browse FTP Server', 'ftp-media'); ?>
                        </button>
                        <button type="button" id="go-back" class="button" style="display: none;">
                            <?php _e('← Back', 'ftp-media'); ?>
                        </button>
                        <span id="current-path" class="ftp-current-path"></span>
                    </div>
                    
                    <div id="browser-results" style="display: none;">
                        <div class="ftp-browser-controls">
                            <label><input type="checkbox" id="select-all-files"> <?php _e('Select All Items', 'ftp-media'); ?></label>
                            <span class="description"><?php _e('Check items to import them, or use "Open" buttons to navigate folders', 'ftp-media'); ?></span>
                            <button type="button" id="select-none-files" class="button"><?php _e('Select None', 'ftp-media'); ?></button>
                        </div>
                        <div id="browser-contents"></div>
                        <button type="button" id="import-selected-files" class="button button-primary" style="display: none;">
                            <?php _e('Import Selected Files', 'ftp-media'); ?>
                        </button>
                    </div>
                </div>
                
                <div id="import-progress" style="display: none;">
                    <h3><?php _e('Import Progress', 'ftp-media'); ?></h3>
                    <div id="import-status"></div>
                </div>
            </div>
            
            <div class="ftp-import-help">
                <h2><?php _e('How to Use', 'ftp-media'); ?></h2>
                <ol>
                    <li><?php _e('Manually upload your old media files to your FTP server maintaining the WordPress directory structure (e.g., /2025/10/filename.jpg)', 'ftp-media'); ?></li>
                    <li><?php _e('Click "Scan FTP Server" to find all files on your FTP server', 'ftp-media'); ?></li>
                    <li><?php _e('Select the files you want to import into WordPress', 'ftp-media'); ?></li>
                    <li><?php _e('Click "Import Selected Files" to add them to your WordPress media library', 'ftp-media'); ?></li>
                </ol>
            </div>
        </div>
        <?php
    }
    
    /**
     * Handle media upload - upload to FTP server and clean up local files
     */
    public function handle_media_upload($metadata, $attachment_id) {
        // Log to plugin directory directly
        $log_file = FTP_MEDIA_PLUGIN_PATH . 'upload-debug.log';
        $log_msg = "\n" . date('Y-m-d H:i:s') . " - ";
        
        file_put_contents($log_file, $log_msg . "==================== FTP MEDIA UPLOAD START ====================", FILE_APPEND);
        file_put_contents($log_file, "\n" . date('Y-m-d H:i:s') . " - handle_media_upload() CALLED with attachment_id: {$attachment_id}", FILE_APPEND);
        
        $settings = get_option('ftp_media_settings');
        
        if (!$settings['enabled']) {
            file_put_contents($log_file, "\n" . date('Y-m-d H:i:s') . " - FTP Media is DISABLED, returning early", FILE_APPEND);
            return $metadata;
        }
        
        file_put_contents($log_file, "\n" . date('Y-m-d H:i:s') . " - FTP Media is ENABLED", FILE_APPEND);
        
        // Get the attachment file path
        $file_path = get_attached_file($attachment_id);
        
        if (!$file_path || !file_exists($file_path)) {
            file_put_contents($log_file, "\n" . date('Y-m-d H:i:s') . " - File not found for attachment ID: {$attachment_id}", FILE_APPEND);
            return $metadata;
        }
        
        // If metadata['file'] is missing (especially for non-image files), let's construct it.
        if (empty($metadata['file'])) {
            $upload_dir = wp_upload_dir();
            $relative_path = str_replace($upload_dir['basedir'] . '/', '', $file_path);
            if ($relative_path !== $file_path) { // Ensure the replace happened
                $metadata['file'] = $relative_path;
                file_put_contents($log_file, "\n" . date('Y-m-d H:i:s') . " - Metadata[file] was empty. Manually constructed: " . $metadata['file'], FILE_APPEND);
            }
        }
        
        file_put_contents($log_file, "\n" . date('Y-m-d H:i:s') . " - Attachment ID: {$attachment_id}", FILE_APPEND);
        file_put_contents($log_file, "\n" . date('Y-m-d H:i:s') . " - Original metadata[file]: " . (isset($metadata['file']) ? $metadata['file'] : 'NOT SET'), FILE_APPEND);
        
        $ftp_uploader = new FTP_Uploader();
        
        // Upload main file
        $result = $ftp_uploader->upload_to_ftp($file_path, wp_get_attachment_url($attachment_id));
        
        file_put_contents($log_file, "\n" . date('Y-m-d H:i:s') . " - FTP upload result success: " . ($result['success'] ? 'YES' : 'NO'), FILE_APPEND);
        file_put_contents($log_file, "\n" . date('Y-m-d H:i:s') . " - FTP URL returned: " . (isset($result['ftp_url']) ? $result['ftp_url'] : 'NO URL'), FILE_APPEND);
        file_put_contents($log_file, "\n" . date('Y-m-d H:i:s') . " - FTP remote_path: " . (isset($result['remote_path']) ? $result['remote_path'] : 'NO PATH'), FILE_APPEND);
        
        if (!$result['success']) {
            file_put_contents($log_file, "\n" . date('Y-m-d H:i:s') . " - FTP upload FAILED: " . $result['message'], FILE_APPEND);
            wp_delete_attachment($attachment_id, true);
            return array();
        }
        
        // Main file uploaded successfully
        file_put_contents($log_file, "\n" . date('Y-m-d H:i:s') . " - FTP upload SUCCESS: " . $result['ftp_url'], FILE_APPEND);
        
        // Store FTP URL in post meta
        update_post_meta($attachment_id, '_ftp_url', $result['ftp_url']);
        
        // Upload all thumbnails to FTP
        $uploaded_thumbnails = array();
        if (isset($metadata['sizes']) && is_array($metadata['sizes'])) {
            $upload_dir = wp_upload_dir();
            $base_dir = dirname($file_path);
            file_put_contents($log_file, "\n" . date('Y-m-d H:i:s') . " - Found " . count($metadata['sizes']) . " thumbnail sizes", FILE_APPEND);
            
            foreach ($metadata['sizes'] as $size => $size_data) {
                if (isset($size_data['file'])) {
                    $thumbnail_file = $base_dir . '/' . $size_data['file'];
                    
                    if (file_exists($thumbnail_file)) {
                        file_put_contents($log_file, "\n" . date('Y-m-d H:i:s') . " - Uploading thumbnail ({$size}): {$thumbnail_file}", FILE_APPEND);
                        
                        $thumb_result = $ftp_uploader->upload_to_ftp($thumbnail_file, '');
                        
                        if ($thumb_result['success']) {
                            file_put_contents($log_file, "\n" . date('Y-m-d H:i:s') . " - Thumbnail ({$size}) uploaded: " . basename($thumb_result['ftp_url']), FILE_APPEND);
                            $uploaded_thumbnails[$size] = basename($thumb_result['ftp_url']);
                        }
                    }
                }
            }
        }
        
        // Check if file was renamed
        file_put_contents($log_file, "\n" . date('Y-m-d H:i:s') . " - ========== CHECKING IF FILE WAS RENAMED ==========", FILE_APPEND);
        file_put_contents($log_file, "\n" . date('Y-m-d H:i:s') . " - Metadata[file] BEFORE: " . $metadata['file'], FILE_APPEND);
        
        if (isset($metadata['file'])) {
            $ftp_filename = basename($result['ftp_url']);
            $original_filename = basename($metadata['file']);
            
            file_put_contents($log_file, "\n" . date('Y-m-d H:i:s') . " - FTP filename: " . $ftp_filename, FILE_APPEND);
            file_put_contents($log_file, "\n" . date('Y-m-d H:i:s') . " - Original filename: " . $original_filename, FILE_APPEND);
            file_put_contents($log_file, "\n" . date('Y-m-d H:i:s') . " - Are they different? " . ($ftp_filename !== $original_filename ? 'YES' : 'NO'), FILE_APPEND);
            
            // If filenames differ, the file was renamed for uniqueness
            if ($ftp_filename !== $original_filename) {
                file_put_contents($log_file, "\n" . date('Y-m-d H:i:s') . " - FILE WAS RENAMED!", FILE_APPEND);
                file_put_contents($log_file, "\n" . date('Y-m-d H:i:s') . " - From: {$original_filename} To: {$ftp_filename}", FILE_APPEND);
                
                $old_file = $metadata['file'];
                $new_relative_path = dirname($metadata['file']) . '/' . $ftp_filename;
                
                // 1. Update the metadata array to be returned
                $metadata['file'] = $new_relative_path;

                // 2. Force-update the database entry that controls the "File name" display
                update_post_meta($attachment_id, '_wp_attached_file', $new_relative_path);
                
                file_put_contents($log_file, "\n" . date('Y-m-d H:i:s') . " - Old metadata[file]: " . $old_file, FILE_APPEND);
                file_put_contents($log_file, "\n" . date('Y-m-d H:i:s') . " - New metadata[file]: " . $metadata['file'], FILE_APPEND);
                file_put_contents($log_file, "\n" . date('Y-m-d H:i:s') . " - Force-updated _wp_attached_file meta to: " . $new_relative_path, FILE_APPEND);
            } else {
                file_put_contents($log_file, "\n" . date('Y-m-d H:i:s') . " - File was NOT renamed", FILE_APPEND);
            }
        }
        
        file_put_contents($log_file, "\n" . date('Y-m-d H:i:s') . " - Metadata[file] AFTER: " . $metadata['file'], FILE_APPEND);
        
        // Update thumbnail filenames in metadata with actual uploaded filenames
        if (!empty($uploaded_thumbnails) && isset($metadata['sizes']) && is_array($metadata['sizes'])) {
            foreach ($metadata['sizes'] as $size => $size_data) {
                if (isset($uploaded_thumbnails[$size])) {
                    $metadata['sizes'][$size]['file'] = $uploaded_thumbnails[$size];
                    file_put_contents($log_file, "\n" . date('Y-m-d H:i:s') . " - Updated thumbnail {$size}: " . $uploaded_thumbnails[$size], FILE_APPEND);
                }
            }
        }
        
        // Clean up local files after successful FTP upload
        $this->cleanup_local_files($attachment_id, $metadata);
        
        file_put_contents($log_file, "\n" . date('Y-m-d H:i:s') . " - FINAL metadata[file] being RETURNED: " . $metadata['file'], FILE_APPEND);
        file_put_contents($log_file, "\n" . date('Y-m-d H:i:s') . " - ==================== FTP MEDIA UPLOAD END ====================\n", FILE_APPEND);
        
        return $metadata;
    }
    
    /**
     * Handle media deletion - delete files from FTP server
     */
    public function handle_media_delete($attachment_id) {
        $settings = get_option('ftp_media_settings');
        
        if (!$settings['enabled']) {
            return;
        }
        
        // Check if this is an FTP-stored file
        $ftp_url = get_post_meta($attachment_id, '_ftp_url', true);
        
        if (!$ftp_url) {
            return; // Not an FTP file, let WordPress handle it normally
        }
        
        error_log('FTP Media: Deleting file from FTP server for attachment ID: ' . $attachment_id);
        
        // Get metadata to find all thumbnail files
        $metadata = wp_get_attachment_metadata($attachment_id);
        
        // Initialize FTP uploader (which has delete methods)
        $ftp_uploader = new FTP_Uploader();
        
        try {
            // Delete main file
            $result = $ftp_uploader->delete_from_ftp($ftp_url);
            
            if ($result['success']) {
                error_log('FTP Media: Successfully deleted main file from FTP: ' . $ftp_url);
            } else {
                error_log('FTP Media: Failed to delete main file from FTP: ' . $result['message']);
            }
            
            // Delete all thumbnails
            if (isset($metadata['sizes']) && is_array($metadata['sizes'])) {
                $ftp_base_url = dirname($ftp_url);
                
                foreach ($metadata['sizes'] as $size => $size_data) {
                    if (isset($size_data['file'])) {
                        $thumbnail_url = $ftp_base_url . '/' . $size_data['file'];
                        
                        $thumb_result = $ftp_uploader->delete_from_ftp($thumbnail_url);
                        
                        if ($thumb_result['success']) {
                            error_log("FTP Media: Successfully deleted thumbnail ({$size}) from FTP: {$thumbnail_url}");
                        } else {
                            error_log("FTP Media: Failed to delete thumbnail ({$size}) from FTP: {$thumb_result['message']}");
                        }
                    }
                }
            }
            
        } catch (Exception $e) {
            error_log('FTP Media: Exception while deleting from FTP: ' . $e->getMessage());
        }
    }
    
    /**
     * Clean up local files after successful FTP upload
     */
    private function cleanup_local_files($attachment_id, $metadata) {
        $upload_dir = wp_upload_dir();
        $file_path = get_attached_file($attachment_id);
        
        // Delete main file
        if ($file_path && file_exists($file_path)) {
            unlink($file_path);
            error_log('FTP Media: Cleaned up local file: ' . $file_path);
        }
        
        // Delete thumbnails if they exist
        if (isset($metadata['sizes']) && is_array($metadata['sizes'])) {
            foreach ($metadata['sizes'] as $size => $size_data) {
                if (isset($size_data['file'])) {
                    $size_file_path = $upload_dir['basedir'] . '/' . dirname($metadata['file']) . '/' . $size_data['file'];
                    if (file_exists($size_file_path)) {
                        unlink($size_file_path);
                        error_log('FTP Media: Cleaned up local thumbnail: ' . $size_file_path);
                    }
                }
            }
        }
        
        // Delete any other generated files
        if (isset($metadata['file'])) {
            $base_file_path = $upload_dir['basedir'] . '/' . $metadata['file'];
            $base_dir = dirname($base_file_path);
            
            // If it's a directory, try to remove it
            if (is_dir($base_dir)) {
                $this->recursive_rmdir($base_dir);
                error_log('FTP Media: Cleaned up directory: ' . $base_dir);
            }
        }
    }
    
    /**
     * Recursively remove directory
     */
    private function recursive_rmdir($dir) {
        if (is_dir($dir)) {
            $objects = scandir($dir);
            foreach ($objects as $object) {
                if ($object != "." && $object != "..") {
                    if (is_dir($dir . "/" . $object)) {
                        $this->recursive_rmdir($dir . "/" . $object);
                    } else {
                        unlink($dir . "/" . $object);
                    }
                }
            }
            rmdir($dir);
        }
    }
    
    /**
     * Validate upload before processing
     */
    public function validate_upload($file) {
        $settings = get_option('ftp_media_settings');
        
        if (!$settings['enabled']) {
            return $file;
        }
        
        // Check if FTP settings are configured
        if (empty($settings['ftp_host']) || empty($settings['ftp_username'])) {
            $file['error'] = 'FTP Media is enabled but FTP settings are not configured. Please configure your FTP settings first.';
            return $file;
        }
        
        return $file;
    }
    
    
    /**
     * Filter attachment URLs to use FTP URLs
     */
    public function filter_attachment_url($url, $attachment_id) {
        $settings = get_option('ftp_media_settings');
        
        if (!$settings['enabled']) {
            return $url;
        }
        
        $ftp_url = get_post_meta($attachment_id, '_ftp_url', true);
        
        if ($ftp_url) {
            return $ftp_url;
        }
        
        return $url;
    }
    
    /**
     * Filter attachment image src to use FTP URLs
     */
    public function filter_attachment_image_src($image, $attachment_id, $size, $icon) {
        $settings = get_option('ftp_media_settings');
        
        if (!$settings['enabled']) {
            return $image;
        }
        
        $ftp_url = get_post_meta($attachment_id, '_ftp_url', true);
        
        if (!$ftp_url) {
            return $image;
        }
        
        // If requesting full size, return the main FTP URL
        if ($size === 'full' || !is_string($size)) {
            if (isset($image[0])) {
                $image[0] = $ftp_url;
            }
            return $image;
        }
        
        // For specific thumbnail sizes, construct the thumbnail URL
        $metadata = wp_get_attachment_metadata($attachment_id);
        
        if (isset($metadata['sizes'][$size]['file'])) {
            // Get the thumbnail filename
            $thumbnail_file = $metadata['sizes'][$size]['file'];
            
            // Get the base FTP URL (directory path)
            $ftp_base_url = dirname($ftp_url);
            
            // Construct the full thumbnail URL
            $thumbnail_url = $ftp_base_url . '/' . $thumbnail_file;
            
            if (isset($image[0])) {
                $image[0] = $thumbnail_url;
            }
            
            // Update width and height if available
            if (isset($metadata['sizes'][$size]['width'])) {
                $image[1] = $metadata['sizes'][$size]['width'];
            }
            if (isset($metadata['sizes'][$size]['height'])) {
                $image[2] = $metadata['sizes'][$size]['height'];
            }
        } else {
            // Fallback to main URL if size not found
            if (isset($image[0])) {
                $image[0] = $ftp_url;
            }
        }
        
        return $image;
    }
    
    /**
     * AJAX handler for testing FTP operations
     */
    public function ajax_test_ftp_operations() {
        try {
            // Verify nonce
            if (!wp_verify_nonce($_POST['nonce'], 'ftp_media_test_connection')) {
                wp_send_json_error('Security check failed');
                return;
            }
            
            // Check user permissions
            if (!current_user_can('manage_options')) {
                wp_send_json_error('Insufficient permissions');
                return;
            }
            
            $ftp_uploader = new FTP_Uploader();
            $result = $ftp_uploader->test_ftp_operations();
            
            if ($result['success']) {
                wp_send_json_success($result['message']);
            } else {
                wp_send_json_error($result['message']);
            }
            
        } catch (Exception $e) {
            wp_send_json_error('Unexpected error: ' . $e->getMessage());
        }
    }
    
    /**
     * AJAX handler for testing FTP connection
     */
    public function ajax_test_ftp_connection() {
        try {
            // Verify nonce
            if (!wp_verify_nonce($_POST['nonce'], 'ftp_media_test_connection')) {
                wp_send_json_error('Security check failed');
                return;
            }
            
            // Check user permissions
            if (!current_user_can('manage_options')) {
                wp_send_json_error('Insufficient permissions');
                return;
            }
            
            // Check if settings data is provided
            if (!isset($_POST['settings']) || !is_array($_POST['settings'])) {
                wp_send_json_error('Invalid settings data');
                return;
            }
            
            // Get settings from POST data
            $settings = array(
                'ftp_host' => sanitize_text_field($_POST['settings']['ftp_host']),
                'ftp_username' => sanitize_text_field($_POST['settings']['ftp_username']),
                'ftp_password' => sanitize_text_field($_POST['settings']['ftp_password']),
                'ftp_port' => absint($_POST['settings']['ftp_port']) ?: 21,
                'ftp_directory' => sanitize_text_field($_POST['settings']['ftp_directory']),
                'ftp_passive' => isset($_POST['settings']['ftp_passive']),
                'ftp_ssl' => false, // Force non-SSL
                'enabled' => true // Enable for testing
            );
            
            // Temporarily update settings for testing
            $original_settings = get_option('ftp_media_settings');
            update_option('ftp_media_settings', $settings);
            
            // Test connection
            $ftp_uploader = new FTP_Uploader();
            $result = $ftp_uploader->test_connection();
            
            // Restore original settings
            update_option('ftp_media_settings', $original_settings);
            
            // Send response
            if ($result['success']) {
                wp_send_json_success($result['message']);
            } else {
                wp_send_json_error($result['message']);
            }
            
        } catch (Exception $e) {
            wp_send_json_error('Unexpected error: ' . $e->getMessage());
        }
    }
    
    /**
     * AJAX handler for scanning FTP files
     */
    public function ajax_scan_ftp_files() {
        try {
            // Log the request
            error_log('FTP Media: AJAX scan_ftp_files called');
            
            // Verify nonce
            if (!wp_verify_nonce($_POST['nonce'], 'ftp_media_test_connection')) {
                error_log('FTP Media: Nonce verification failed');
                wp_send_json_error('Security check failed');
                return;
            }
            
            // Check user permissions
            if (!current_user_can('manage_options')) {
                error_log('FTP Media: User permission check failed');
                wp_send_json_error('Insufficient permissions');
                return;
            }
            
            error_log('FTP Media: Creating FTP_Importer instance');
            $importer = new FTP_Importer();
            
            // Get directory parameter
            $directory = isset($_POST['directory']) ? sanitize_text_field($_POST['directory']) : '';
            
            error_log('FTP Media: Starting FTP directory scan for: ' . $directory);
            $result = $importer->get_directory_contents($directory);
            
            error_log('FTP Media: Scan result: ' . print_r($result, true));
            
            if ($result['success']) {
                wp_send_json_success($result);
            } else {
                wp_send_json_error($result['message']);
            }
            
        } catch (Exception $e) {
            error_log('FTP Media: Exception in ajax_scan_ftp_files: ' . $e->getMessage());
            error_log('FTP Media: Exception trace: ' . $e->getTraceAsString());
            wp_send_json_error('Unexpected error: ' . $e->getMessage());
        }
    }
    
    /**
     * AJAX handler for importing FTP files
     */
    public function ajax_import_ftp_files() {
        try {
            // Verify nonce
            if (!wp_verify_nonce($_POST['nonce'], 'ftp_media_test_connection')) {
                wp_send_json_error('Security check failed');
                return;
            }
            
            // Check user permissions
            if (!current_user_can('manage_options')) {
                wp_send_json_error('Insufficient permissions');
                return;
            }
            
            if (!isset($_POST['files']) || !is_array($_POST['files'])) {
                wp_send_json_error('No files selected for import');
                return;
            }
            
            $importer = new FTP_Importer();
            $result = $importer->import_files($_POST['files']);
            
            if ($result['success']) {
                wp_send_json_success($result);
            } else {
                wp_send_json_error($result['message']);
            }
            
        } catch (Exception $e) {
            wp_send_json_error('Unexpected error: ' . $e->getMessage());
        }
    }
    
    /**
     * AJAX handler for migrating FTP URLs to local
     */
    public function ajax_migrate_ftp_to_local() {
        try {
            // Verify nonce
            if (!wp_verify_nonce($_POST['nonce'], 'ftp_media_test_connection')) {
                wp_send_json_error('Security check failed');
                return;
            }
            
            // Check user permissions
            if (!current_user_can('manage_options')) {
                wp_send_json_error('Insufficient permissions');
                return;
            }
            
            // Get all attachments with FTP URLs
            global $wpdb;
            
            $ftp_attachments = $wpdb->get_results($wpdb->prepare(
                "SELECT p.ID, p.post_title, pm.meta_value as ftp_url 
                 FROM {$wpdb->posts} p 
                 INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id 
                 WHERE p.post_type = 'attachment' 
                 AND pm.meta_key = '_ftp_url' 
                 AND pm.meta_value IS NOT NULL 
                 AND pm.meta_value != ''"
            ));
            
            if (empty($ftp_attachments)) {
                wp_send_json_error('No FTP attachments found to migrate.');
                return;
            }
            
            $updated = 0;
            $failed = 0;
            $upload_dir = wp_upload_dir();
            
            foreach ($ftp_attachments as $attachment) {
                try {
                    // Get FTP URL
                    $ftp_url = $attachment->ftp_url;
                    
                    // Extract relative path from FTP URL
                    $relative_path = $this->extract_relative_path_from_ftp_url($ftp_url);
                    
                    if (!$relative_path) {
                        $failed++;
                        continue;
                    }
                    
                    // Construct local file path
                    $local_file_path = $upload_dir['basedir'] . '/' . $relative_path;
                    
                    // Check if file exists locally
                    if (!file_exists($local_file_path)) {
                        $failed++;
                        continue;
                    }
                    
                    // Generate local URL
                    $local_url = $upload_dir['baseurl'] . '/' . $relative_path;
                    
                    // Update attachment metadata
                    $metadata = wp_get_attachment_metadata($attachment->ID);
                    
                    // Update the file path in metadata
                    if ($metadata) {
                        $metadata['file'] = $relative_path;
                        wp_update_attachment_metadata($attachment->ID, $metadata);
                    }
                    
                    // Update the attached file meta
                    update_post_meta($attachment->ID, '_wp_attached_file', $relative_path);
                    
                    // Remove the FTP URL meta (or keep it for reference, your choice)
                    // For now, we'll remove it to mark the migration as complete
                    delete_post_meta($attachment->ID, '_ftp_url');
                    
                    // Update all thumbnail sizes if they exist
                    if (isset($metadata['sizes']) && is_array($metadata['sizes'])) {
                        foreach ($metadata['sizes'] as $size => $size_data) {
                            if (isset($size_data['file'])) {
                                $thumbnail_path = dirname($local_file_path) . '/' . $size_data['file'];
                                
                                // Check if thumbnail exists locally
                                if (file_exists($thumbnail_path)) {
                                    // Thumbnail exists, no need to update metadata
                                }
                            }
                        }
                    }
                    
                    $updated++;
                    
                } catch (Exception $e) {
                    error_log('Error migrating attachment ID ' . $attachment->ID . ': ' . $e->getMessage());
                    $failed++;
                }
            }
            
            wp_send_json_success(array(
                'message' => sprintf(
                    'Migration completed. Updated: %d, Failed: %d',
                    $updated,
                    $failed
                ),
                'updated' => $updated,
                'failed' => $failed
            ));
            
        } catch (Exception $e) {
            wp_send_json_error('Unexpected error: ' . $e->getMessage());
        }
    }
    
    /**
     * Extract relative path from FTP URL
     */
    private function extract_relative_path_from_ftp_url($ftp_url) {
        $settings = get_option('ftp_media_settings');
        
        if (empty($settings['ftp_url'])) {
            return null;
        }
        
        $ftp_base_url = rtrim($settings['ftp_url'], '/');
        $directory = trim($settings['ftp_directory'], '/');
        
        // Build the base FTP path
        if ($directory && $directory !== '/') {
            $ftp_base_url .= '/' . $directory;
        }
        
        // Extract relative path
        $relative_path = str_replace($ftp_base_url . '/', '', $ftp_url);
        
        return $relative_path;
    }
    
    
}
