<?php
/**
 * FTP Importer Class
 * 
 * Handles scanning FTP server and importing files to WordPress database
 * 
 * @package FTP_Media
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class FTP_Importer {
    
    private $settings;
    private $logger;
    private $connection;
    
    public function __construct() {
        $this->settings = get_option('ftp_media_settings', array());
        $this->logger = new FTP_Logger();
    }
    
    /**
     * Scan FTP server for files
     */
    public function scan_ftp_files($directory = '') {
        $this->logger->info("Starting FTP scan for directory: {$directory}");
        
        try {
            // Check if FTP settings are configured
            if (empty($this->settings['ftp_host']) || empty($this->settings['ftp_username'])) {
                throw new Exception('FTP settings are not configured. Please configure your FTP settings first.');
            }
            
            // Connect to FTP server
            $this->connect();
            
            if (!$this->connection) {
                throw new Exception('Failed to connect to FTP server');
            }
            
            $this->logger->info("Connected to FTP server, testing basic operations");
            
            // Test basic FTP operations
            $current_dir = @ftp_pwd($this->connection);
            if ($current_dir === false) {
                throw new Exception('Could not get current directory from FTP server');
            }
            
            $this->logger->info("Current FTP directory: {$current_dir}");
            
            // Set a reasonable timeout for the scan operation
            set_time_limit(300); // 5 minutes
            
            $this->logger->info("Starting directory scan");
            $files = $this->scan_directory($directory);
            
            $this->disconnect();
            
            $this->logger->info("FTP scan completed. Found " . count($files) . " files");
            
            return array(
                'success' => true,
                'files' => $files,
                'count' => count($files)
            );
            
        } catch (Exception $e) {
            $this->logger->error("FTP scan failed: " . $e->getMessage());
            $this->disconnect();
            
            return array(
                'success' => false,
                'message' => $e->getMessage()
            );
        }
    }
    
    /**
     * Get directory contents (files and folders) for browsing
     */
    public function get_directory_contents($directory = '') {
        $this->logger->info("Getting directory contents for: {$directory}");
        
        try {
            // Check if FTP settings are configured
            if (empty($this->settings['ftp_host']) || empty($this->settings['ftp_username'])) {
                throw new Exception('FTP settings are not configured. Please configure your FTP settings first.');
            }
            
            // Connect to FTP server
            $this->connect();
            
            if (!$this->connection) {
                throw new Exception('Failed to connect to FTP server');
            }
            
            $this->logger->info("Connected to FTP server, getting directory contents");
            
            // Test basic FTP operations
            $current_dir = @ftp_pwd($this->connection);
            if ($current_dir === false) {
                throw new Exception('Could not get current directory from FTP server');
            }
            
            $this->logger->info("Current FTP directory: {$current_dir}");
            
            // Get directory contents
            $contents = $this->list_directory_contents($directory);
            
            $this->disconnect();
            
            $this->logger->info("Directory scan completed. Found " . count($contents) . " items");
            
            return array(
                'success' => true,
                'contents' => $contents,
                'current_directory' => $directory,
                'count' => count($contents)
            );
            
        } catch (Exception $e) {
            $this->logger->error("Directory scan failed: " . $e->getMessage());
            $this->disconnect();
            
            return array(
                'success' => false,
                'message' => $e->getMessage()
            );
        }
    }
    
    /**
     * List directory contents (non-recursive)
     */
    private function list_directory_contents($directory) {
        $contents = array();
        
        try {
            $this->logger->info("Listing directory: {$directory}");
            
            // Get directory listing with timeout - try both methods
            $listing = @ftp_nlist($this->connection, $directory);
            $raw_listing = null;
            
            if ($listing === false) {
                $this->logger->warning("ftp_nlist failed, trying ftp_rawlist for directory: {$directory}");
                $raw_listing = @ftp_rawlist($this->connection, $directory);
                
                if ($raw_listing === false) {
                    $this->logger->warning("Could not list directory: {$directory}");
                    return $contents;
                }
                
                // Parse raw listing to get just the filenames
                $listing = array();
                foreach ($raw_listing as $line) {
                    $parts = preg_split('/\s+/', $line, 9);
                    if (count($parts) >= 9) {
                        $filename = $parts[8];
                        // Skip system entries and hidden files
                        if ($filename !== '.' && $filename !== '..' && strpos($filename, '.') !== 0) {
                            $listing[] = $filename;
                        }
                    }
                }
            } else {
                // Try to get raw listing for better directory detection
                $raw_listing = @ftp_rawlist($this->connection, $directory);
            }
            
            if (empty($listing)) {
                $this->logger->info("Directory is empty: {$directory}");
                return $contents;
            }
            
            $this->logger->info("Found " . count($listing) . " items in directory: {$directory}");
            $this->logger->info("Raw listing items: " . print_r($listing, true));
            
            foreach ($listing as $item) {
                // Clean up the item name - remove any directory prefix
                $clean_item = basename($item);
                
                // Skip hidden files and directories (starting with .)
                if (strpos($clean_item, '.') === 0) {
                    continue;
                }
                
                // Skip system entries
                if ($clean_item === '.' || $clean_item === '..') {
                    continue;
                }
                
                // Build the full path properly
                $full_path = $directory ? $directory . '/' . $clean_item : $clean_item;
                
                try {
                    $this->logger->info("Processing item: {$item} at path: {$full_path}");
                    
                    // Check if it's a directory using multiple methods
                    $is_directory = $this->is_directory($full_path, $raw_listing);
                    
                    $this->logger->info("Directory check result for {$full_path}: " . ($is_directory ? 'true' : 'false'));
                    
                    if ($is_directory) {
                        $this->logger->info("Found directory: {$full_path}");
                        $contents[] = array(
                            'type' => 'directory',
                            'path' => $full_path,
                            'name' => $clean_item,
                            'size' => 0,
                            'date' => time(),
                            'icon' => '📁'
                        );
                    } else {
                        // It's a file
                        $file_size = @ftp_size($this->connection, $full_path);
                        $file_date = @ftp_mdtm($this->connection, $full_path);
                        
                        $contents[] = array(
                            'type' => 'file',
                            'path' => $full_path,
                            'name' => $clean_item,
                            'size' => $file_size !== false ? $file_size : 0,
                            'date' => $file_date !== false ? $file_date : time(),
                            'icon' => $this->get_file_icon($clean_item)
                        );
                        
                        $this->logger->info("Found file: {$full_path} (Size: {$file_size})");
                    }
                } catch (Exception $e) {
                    $this->logger->warning("Error processing item {$full_path}: " . $e->getMessage());
                    continue;
                }
            }
            
        } catch (Exception $e) {
            $this->logger->error("Error listing directory {$directory}: " . $e->getMessage());
        }
        
        return $contents;
    }
    
    /**
     * Get file icon based on extension
     */
    private function get_file_icon($filename) {
        $extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        
        $icons = array(
            'jpg' => '🖼️',
            'jpeg' => '🖼️',
            'png' => '🖼️',
            'gif' => '🖼️',
            'webp' => '🖼️',
            'pdf' => '📄',
            'doc' => '📄',
            'docx' => '📄',
            'txt' => '📄',
            'zip' => '📦',
            'rar' => '📦',
            'mp4' => '🎥',
            'avi' => '🎥',
            'mov' => '🎥',
            'mp3' => '🎵',
            'wav' => '🎵'
        );
        
        return isset($icons[$extension]) ? $icons[$extension] : '📄';
    }
    
    /**
     * Check if path is a directory
     */
    private function is_directory($path, $raw_listing = null) {
        try {
            // Method 1: Use raw listing if available (most reliable)
            if ($raw_listing !== null) {
                $filename = basename($path);
                foreach ($raw_listing as $line) {
                    $parts = preg_split('/\s+/', $line, 9);
                    if (count($parts) >= 9 && $parts[8] === $filename) {
                        // Check if it starts with 'd' (directory) in the permissions
                        if (strpos($parts[0], 'd') === 0) {
                            $this->logger->info("Directory check by raw listing: {$path} is a directory");
                            return true;
                        } else {
                            $this->logger->info("Directory check by raw listing: {$path} is not a directory");
                            return false;
                        }
                    }
                }
            }
            
            // Method 2: Try to change to the directory
            $current_dir = @ftp_pwd($this->connection);
            if ($current_dir === false) {
                $this->logger->warning("Could not get current directory");
                return false;
            }
            
            $changed = @ftp_chdir($this->connection, $path);
            
            if ($changed) {
                // Change back to original directory
                @ftp_chdir($this->connection, $current_dir);
                $this->logger->info("Directory check successful: {$path} is a directory");
                return true;
            }
            
            // Method 3: Try to get file size (directories usually return -1 or false)
            $size = @ftp_size($this->connection, $path);
            if ($size === -1) {
                $this->logger->info("Directory check by size: {$path} appears to be a directory (size: -1)");
                return true;
            }
            
            // Method 4: Try to get modification time (some servers return false for directories)
            $mtime = @ftp_mdtm($this->connection, $path);
            if ($mtime === false && $size === false) {
                $this->logger->info("Directory check by mtime: {$path} appears to be a directory (no mtime)");
                return true;
            }
            
            $this->logger->info("Directory check failed: {$path} is not a directory (size: {$size}, mtime: {$mtime})");
            return false;
            
        } catch (Exception $e) {
            $this->logger->warning("Error checking if path is directory {$path}: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Connect to FTP server
     */
    private function connect() {
        try {
            $this->logger->info("Attempting to connect to FTP server: {$this->settings['ftp_host']}:{$this->settings['ftp_port']} (SSL: " . ($this->settings['ftp_ssl'] ? 'Yes' : 'No') . ")");
            
            // Set connection timeout
            $timeout = 30;
            
            // Create FTP connection
            if ($this->settings['ftp_ssl']) {
                $this->connection = @ftp_ssl_connect($this->settings['ftp_host'], $this->settings['ftp_port'], $timeout);
            } else {
                $this->connection = @ftp_connect($this->settings['ftp_host'], $this->settings['ftp_port'], $timeout);
            }
            
            if (!$this->connection) {
                throw new Exception('Could not connect to FTP server. Please check your host and port settings.');
            }
            
            $this->logger->info("FTP connection established, attempting login");
            
            // Login to FTP server
            $login_result = @ftp_login($this->connection, $this->settings['ftp_username'], $this->settings['ftp_password']);
            
            if (!$login_result) {
                throw new Exception('Could not login to FTP server. Please check your username and password.');
            }
            
            $this->logger->info("FTP login successful");
            
            // Set passive mode if enabled
            if ($this->settings['ftp_passive']) {
                @ftp_pasv($this->connection, true);
                $this->logger->info("Passive mode enabled");
            }
            
            // Change to specified directory
            if (!empty($this->settings['ftp_directory']) && $this->settings['ftp_directory'] !== '/') {
                $directory = rtrim($this->settings['ftp_directory'], '/');
                $this->logger->info("Attempting to change to directory: {$directory}");
                
                if (!@ftp_chdir($this->connection, $directory)) {
                    throw new Exception("Could not change to directory: {$directory}. Please check your directory path.");
                }
                
                $this->logger->info("Successfully changed to directory: {$directory}");
            }
            
            $this->logger->info('FTP connection established successfully');
            
        } catch (Exception $e) {
            $this->logger->error("FTP connection failed: " . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Disconnect from FTP server
     */
    private function disconnect() {
        if ($this->connection) {
            ftp_close($this->connection);
            $this->connection = null;
        }
    }
    
    /**
     * Import files and folders to WordPress database
     */
    public function import_files($items_to_import) {
        $this->logger->info("Starting import of " . count($items_to_import) . " items");
        
        $imported = 0;
        $skipped = 0;
        $errors = 0;
        
        foreach ($items_to_import as $item_data) {
            try {
                if ($item_data['type'] === 'directory') {
                    // Import folder recursively
                    $result = $this->import_folder_recursive($item_data['path']);
                    $imported += $result['imported'];
                    $skipped += $result['skipped'];
                    $errors += $result['errors'];
                } else {
                    // Import single file
                    $result = $this->import_single_file($item_data);
                    
                    if ($result['success']) {
                        $imported++;
                        $this->logger->info("Imported file: " . $item_data['path']);
                    } else {
                        if (isset($result['skipped']) && $result['skipped']) {
                            $skipped++;
                            $this->logger->info("Skipped existing file: " . $item_data['path']);
                        } else {
                            $errors++;
                            $this->logger->error("Error importing file: " . $item_data['path'] . " - " . $result['message']);
                        }
                    }
                }
                
            } catch (Exception $e) {
                $errors++;
                $this->logger->error("Error importing item: " . $item_data['path'] . " - " . $e->getMessage());
            }
        }
        
        $this->logger->info("Import completed. Imported: {$imported}, Skipped: {$skipped}, Errors: {$errors}");
        
        return array(
            'success' => true,
            'imported' => $imported,
            'skipped' => $skipped,
            'errors' => $errors
        );
    }
    
    /**
     * Import folder recursively
     */
    private function import_folder_recursive($folder_path) {
        $this->logger->info("Importing folder recursively: {$folder_path}");
        
        $imported = 0;
        $skipped = 0;
        $errors = 0;
        
        try {
            // Connect to FTP server
            $this->connect();
            
            if (!$this->connection) {
                throw new Exception('Failed to connect to FTP server');
            }
            
            // Get all files in the folder recursively
            $files = $this->scan_directory_recursive($folder_path);
            
            $this->logger->info("Found " . count($files) . " files to process in folder: {$folder_path}");
            
            $this->disconnect();
            
            // Import each file
            foreach ($files as $file_data) {
                try {
                    $this->logger->info("Processing file for import: " . $file_data['path']);
                    $result = $this->import_single_file($file_data);
                    
                    if ($result['success']) {
                        $imported++;
                        $this->logger->info("Imported file: " . $file_data['path']);
                    } else {
                        if (isset($result['skipped']) && $result['skipped']) {
                            $skipped++;
                            $this->logger->info("Skipped existing file: " . $file_data['path']);
                        } else {
                            $errors++;
                            $this->logger->error("Error importing file: " . $file_data['path'] . " - " . $result['message']);
                        }
                    }
                    
                } catch (Exception $e) {
                    $errors++;
                    $this->logger->error("Error importing file: " . $file_data['path'] . " - " . $e->getMessage());
                }
            }
            
        } catch (Exception $e) {
            $this->logger->error("Error importing folder {$folder_path}: " . $e->getMessage());
            $errors++;
        }
        
        $this->logger->info("Folder import completed for {$folder_path}. Imported: {$imported}, Skipped: {$skipped}, Errors: {$errors}");
        
        return array(
            'imported' => $imported,
            'skipped' => $skipped,
            'errors' => $errors
        );
    }
    
    /**
     * Recursively scan directory for files only
     */
    private function scan_directory_recursive($directory) {
        $files = array();
        
        try {
            $this->logger->info("Scanning directory recursively: {$directory}");
            
            // Get directory listing
            $listing = @ftp_nlist($this->connection, $directory);
            
            if ($listing === false) {
                $this->logger->warning("Could not list directory: {$directory}");
                return $files;
            }
            
            if (empty($listing)) {
                $this->logger->info("Directory is empty: {$directory}");
                return $files;
            }
            
            $this->logger->info("Raw listing for {$directory}: " . print_r($listing, true));
            
            foreach ($listing as $item) {
                // Clean up the item name - remove any directory prefix
                $clean_item = basename($item);
                
                // Skip hidden files and directories (starting with .)
                if (strpos($clean_item, '.') === 0) {
                    continue;
                }
                
                // Skip system entries
                if ($clean_item === '.' || $clean_item === '..') {
                    continue;
                }
                
                $full_path = $directory ? $directory . '/' . $clean_item : $clean_item;
                
                try {
                    $this->logger->info("Processing item: {$clean_item} at path: {$full_path}");
                    
                    // Check if it's a directory
                    $is_directory = $this->is_directory($full_path);
                    
                    if ($is_directory) {
                        $this->logger->info("Found directory, scanning recursively: {$full_path}");
                        // Recursively scan subdirectory
                        $sub_files = $this->scan_directory_recursive($full_path);
                        $files = array_merge($files, $sub_files);
                    } else {
                        // It's a file
                        $file_size = @ftp_size($this->connection, $full_path);
                        $file_date = @ftp_mdtm($this->connection, $full_path);
                        
                        $files[] = array(
                            'path' => $full_path,
                            'name' => $clean_item,
                            'size' => $file_size !== false ? $file_size : 0,
                            'date' => $file_date !== false ? $file_date : time(),
                            'type' => 'file'
                        );
                        
                        $this->logger->info("Found file: {$full_path} (Size: {$file_size})");
                    }
                } catch (Exception $e) {
                    $this->logger->warning("Error processing item {$full_path}: " . $e->getMessage());
                    continue;
                }
            }
            
        } catch (Exception $e) {
            $this->logger->error("Error scanning directory {$directory}: " . $e->getMessage());
        }
        
        $this->logger->info("Recursive scan completed for {$directory}. Found " . count($files) . " files.");
        return $files;
    }
    
    /**
     * Import a single file to WordPress database
     */
    private function import_single_file($file_data) {
        $ftp_path = $file_data['path'];
        $filename = $file_data['name'];
        
        // Generate FTP URL
        $ftp_url = $this->get_ftp_url($ftp_path);
        
        $this->logger->info("Checking for existing attachment: {$filename} at {$ftp_url}");
        
        // Check if file already exists in database
        $existing_attachment = $this->find_existing_attachment($filename, $ftp_path);
        
        if ($existing_attachment) {
            $this->logger->info("File already exists, skipping: {$filename}");
            return array(
                'success' => false,
                'message' => 'File already exists in database',
                'skipped' => true
            );
        } else {
            $this->logger->info("Creating new attachment for: {$filename}");
            // Create new attachment
            return $this->create_new_attachment($file_data, $ftp_url);
        }
    }
    
    /**
     * Find existing attachment by filename and path
     */
    private function find_existing_attachment($filename, $ftp_path) {
        global $wpdb;
        
        // Generate FTP URL for comparison
        $ftp_url = $this->get_ftp_url($ftp_path);
        
        // Try to find by FTP URL first (most reliable)
        $attachment = $wpdb->get_row($wpdb->prepare(
            "SELECT p.* FROM {$wpdb->posts} p 
             INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id 
             WHERE p.post_type = 'attachment' 
             AND pm.meta_key = '_ftp_url' 
             AND pm.meta_value = %s",
            $ftp_url
        ));
        
        if ($attachment) {
            $this->logger->info("Found existing attachment by FTP URL: {$ftp_url}");
            return $attachment;
        }
        
        // Try to find by filename
        $attachment = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$wpdb->posts} WHERE post_type = 'attachment' AND post_title = %s",
            $filename
        ));
        
        if ($attachment) {
            $this->logger->info("Found existing attachment by filename: {$filename}");
            return $attachment;
        }
        
        // Try to find by meta value (original filename)
        $attachment = $wpdb->get_row($wpdb->prepare(
            "SELECT p.* FROM {$wpdb->posts} p 
             INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id 
             WHERE p.post_type = 'attachment' 
             AND pm.meta_key = '_wp_attached_file' 
             AND pm.meta_value LIKE %s",
            '%' . $filename
        ));
        
        if ($attachment) {
            $this->logger->info("Found existing attachment by attached file: {$filename}");
        }
        
        return $attachment;
    }
    
    /**
     * Update existing attachment URL
     */
    private function update_attachment_url($attachment, $ftp_url) {
        // Update the attachment URL in postmeta
        update_post_meta($attachment->ID, '_wp_attached_file', $this->get_relative_path_from_ftp_url($ftp_url));
        
        // Store FTP URL for the plugin to use
        update_post_meta($attachment->ID, '_ftp_url', $ftp_url);
        
        return array(
            'success' => true,
            'message' => 'Updated existing attachment'
        );
    }
    
    /**
     * Create new attachment entry
     */
    private function create_new_attachment($file_data, $ftp_url) {
        // Create attachment post
        $attachment_data = array(
            'post_title' => $file_data['name'],
            'post_content' => '',
            'post_status' => 'inherit',
            'post_parent' => 0,
            'post_type' => 'attachment',
            'post_mime_type' => $this->get_mime_type($file_data['name'])
        );
        
        $attachment_id = wp_insert_post($attachment_data);
        
        if (!$attachment_id) {
            return array(
                'success' => false,
                'message' => 'Failed to create attachment post'
            );
        }
        
        // Add attachment metadata
        update_post_meta($attachment_id, '_wp_attached_file', $this->get_relative_path_from_ftp_url($ftp_url));
        update_post_meta($attachment_id, '_ftp_url', $ftp_url);
        update_post_meta($attachment_id, '_wp_attachment_metadata', array(
            'file' => $this->get_relative_path_from_ftp_url($ftp_url),
            'width' => 0,
            'height' => 0
        ));
        
        return array(
            'success' => true,
            'message' => 'Created new attachment',
            'attachment_id' => $attachment_id
        );
    }
    
    /**
     * Get relative path from FTP URL
     */
    private function get_relative_path_from_ftp_url($ftp_url) {
        $base_url = rtrim($this->settings['ftp_url'], '/');
        $directory = trim($this->settings['ftp_directory'], '/');
        
        if ($directory && $directory !== '/') {
            $base_url .= '/' . $directory;
        }
        
        return str_replace($base_url . '/', '', $ftp_url);
    }
    
    /**
     * Get MIME type for file
     */
    private function get_mime_type($filename) {
        $extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        
        $mime_types = array(
            'jpg' => 'image/jpeg',
            'jpeg' => 'image/jpeg',
            'png' => 'image/png',
            'gif' => 'image/gif',
            'webp' => 'image/webp',
            'pdf' => 'application/pdf',
            'doc' => 'application/msword',
            'docx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'txt' => 'text/plain'
        );
        
        return isset($mime_types[$extension]) ? $mime_types[$extension] : 'application/octet-stream';
    }
    
    /**
     * Get FTP URL for file
     */
    private function get_ftp_url($relative_path) {
        $base_url = rtrim($this->settings['ftp_url'], '/');
        $directory = trim($this->settings['ftp_directory'], '/');
        
        if ($directory && $directory !== '/') {
            return $base_url . '/' . $directory . '/' . $relative_path;
        }
        
        return $base_url . '/' . $relative_path;
    }
    
    /**
     * Get import statistics
     */
    public function get_import_stats() {
        global $wpdb;
        
        $total_attachments = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = 'attachment'");
        
        // Count FTP attachments (those with _ftp_url meta key)
        $ftp_attachments = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->postmeta} pm 
             INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID 
             WHERE p.post_type = 'attachment' 
             AND pm.meta_key = '_ftp_url' 
             AND pm.meta_value IS NOT NULL 
             AND pm.meta_value != ''"
        ));
        
        // Count local attachments (those without _ftp_url or with empty _ftp_url)
        $local_attachments = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->posts} p
             LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_ftp_url'
             WHERE p.post_type = 'attachment' 
             AND (pm.meta_value IS NULL OR pm.meta_value = '')"
        ));
        
        return array(
            'total_attachments' => (int) $total_attachments,
            'ftp_attachments' => (int) $ftp_attachments,
            'local_attachments' => (int) $local_attachments
        );
    }
}
