<?php
/**
 * FTP Media Logger Class
 */

if (!defined('ABSPATH')) {
    exit;
}

class FTP_Logger {
    
    private $log_file;
    
    public function __construct() {
        $upload_dir = wp_upload_dir();
        $this->log_file = $upload_dir['basedir'] . '/ftp-media.log';
    }
    
    /**
     * Log message to file
     */
    public function log($message, $level = 'INFO') {
        $timestamp = current_time('Y-m-d H:i:s');
        $log_entry = "[{$timestamp}] [{$level}] {$message}" . PHP_EOL;
        
        // Write to log file
        file_put_contents($this->log_file, $log_entry, FILE_APPEND | LOCK_EX);
        
        // Also log to WordPress debug log if enabled
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log("FTP Media: {$message}");
        }
    }
    
    /**
     * Log error message
     */
    public function error($message) {
        $this->log($message, 'ERROR');
    }
    
    /**
     * Log warning message
     */
    public function warning($message) {
        $this->log($message, 'WARNING');
    }
    
    /**
     * Log info message
     */
    public function info($message) {
        $this->log($message, 'INFO');
    }
    
    /**
     * Log debug message
     */
    public function debug($message) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            $this->log($message, 'DEBUG');
        }
    }
    
    /**
     * Get log file path
     */
    public function get_log_file() {
        return $this->log_file;
    }
    
    /**
     * Clear log file
     */
    public function clear_log() {
        if (file_exists($this->log_file)) {
            file_put_contents($this->log_file, '');
        }
    }
    
    /**
     * Get log file size
     */
    public function get_log_size() {
        if (file_exists($this->log_file)) {
            return filesize($this->log_file);
        }
        return 0;
    }
    
    /**
     * Get recent log entries
     */
    public function get_recent_entries($lines = 50) {
        if (!file_exists($this->log_file)) {
            return array();
        }
        
        $file = new SplFileObject($this->log_file);
        $file->seek(PHP_INT_MAX);
        $total_lines = $file->key();
        
        $start_line = max(0, $total_lines - $lines);
        $file->seek($start_line);
        
        $entries = array();
        while (!$file->eof()) {
            $line = trim($file->fgets());
            if (!empty($line)) {
                $entries[] = $line;
            }
        }
        
        return $entries;
    }
}
