<?php
/**
 * FTP Media Settings Class
 */

if (!defined('ABSPATH')) {
    exit;
}

class FTP_Settings {
    
    public function __construct() {
        add_action('admin_init', array($this, 'init_settings'));
    }
    
    public function init_settings() {
        register_setting('ftp_media_settings', 'ftp_media_settings', array($this, 'validate_settings'));
        
        add_settings_section(
            'ftp_media_main',
            __('FTP Configuration', 'ftp-media'),
            array($this, 'main_section_callback'),
            'ftp-media-settings'
        );
        
        add_settings_field(
            'enabled',
            __('Enable FTP Media', 'ftp-media'),
            array($this, 'enabled_callback'),
            'ftp-media-settings',
            'ftp_media_main'
        );
        
        add_settings_field(
            'ftp_host',
            __('FTP Host', 'ftp-media'),
            array($this, 'ftp_host_callback'),
            'ftp-media-settings',
            'ftp_media_main'
        );
        
        add_settings_field(
            'ftp_username',
            __('FTP Username', 'ftp-media'),
            array($this, 'ftp_username_callback'),
            'ftp-media-settings',
            'ftp_media_main'
        );
        
        add_settings_field(
            'ftp_password',
            __('FTP Password', 'ftp-media'),
            array($this, 'ftp_password_callback'),
            'ftp-media-settings',
            'ftp_media_main'
        );
        
        add_settings_field(
            'ftp_port',
            __('FTP Port', 'ftp-media'),
            array($this, 'ftp_port_callback'),
            'ftp-media-settings',
            'ftp_media_main'
        );
        
        add_settings_field(
            'ftp_directory',
            __('FTP Directory', 'ftp-media'),
            array($this, 'ftp_directory_callback'),
            'ftp-media-settings',
            'ftp_media_main'
        );
        
        add_settings_field(
            'ftp_url',
            __('FTP URL', 'ftp-media'),
            array($this, 'ftp_url_callback'),
            'ftp-media-settings',
            'ftp_media_main'
        );
        
        add_settings_field(
            'ftp_passive',
            __('Passive Mode', 'ftp-media'),
            array($this, 'ftp_passive_callback'),
            'ftp-media-settings',
            'ftp_media_main'
        );
        
    }
    
    public function render_settings_page() {
        ?>
        <div class="wrap ftp-media-settings">
            <h1><?php _e('FTP Media Settings', 'ftp-media'); ?></h1>
            
            <?php if (isset($_GET['settings-updated']) && $_GET['settings-updated']): ?>
                <div class="notice notice-success is-dismissible">
                    <p><?php _e('Settings saved successfully!', 'ftp-media'); ?></p>
                </div>
            <?php endif; ?>
            
            <form method="post" action="options.php">
                <?php
                settings_fields('ftp_media_settings');
                do_settings_sections('ftp-media-settings');
                submit_button();
                ?>
            </form>
            
                <div class="ftp-test-connection">
                <h3><?php _e('Test FTP Connection', 'ftp-media'); ?></h3>
                <button type="button" id="test-ftp-connection" class="button button-secondary">
                    <?php _e('Test Connection', 'ftp-media'); ?>
                </button>
                <button type="button" id="test-ftp-operations" class="button button-secondary">
                    <?php _e('Test FTP Operations', 'ftp-media'); ?>
                </button>
                <div id="ftp-test-result"></div>
                
                <div class="ftp-troubleshooting">
                    <h4><?php _e('Troubleshooting Tips', 'ftp-media'); ?></h4>
                    <div id="troubleshooting-content">
                        <ul>
                            <li><strong><?php _e('Connection Issues:', 'ftp-media'); ?></strong> <?php _e('Verify your FTP host, port, and credentials are correct.', 'ftp-media'); ?></li>
                            <li><strong><?php _e('SSL/FTPS:', 'ftp-media'); ?></strong> <?php _e('If using SSL, ensure your server supports FTPS and try port 21 with SSL disabled first.', 'ftp-media'); ?></li>
                            <li><strong><?php _e('Firewall:', 'ftp-media'); ?></strong> <?php _e('Check if your hosting provider allows outbound FTP connections.', 'ftp-media'); ?></li>
                            <li><strong><?php _e('Directory:', 'ftp-media'); ?></strong> <?php _e('Ensure the FTP directory exists and you have write permissions.', 'ftp-media'); ?></li>
                            <li><strong><?php _e('Passive Mode:', 'ftp-media'); ?></strong> <?php _e('Try enabling/disabling passive mode if connection fails.', 'ftp-media'); ?></li>
                        </ul>
                    </div>
                    <p><strong><?php _e('Need Help?', 'ftp-media'); ?></strong> <?php _e('Check the plugin logs at:', 'ftp-media'); ?> <code><?php echo wp_upload_dir()['basedir']; ?>/ftp-media.log</code></p>
                </div>
            </div>
            
            <div class="ftp-migrate-to-local">
                <h3><?php _e('Migrate from FTP to Local Host', 'ftp-media'); ?></h3>
                <p><?php _e('If you want to stop using FTP and move your media files back to your local server, follow these steps:', 'ftp-media'); ?></p>
                <ol>
                    <li><?php _e('First, manually download all your media files from your FTP server to your local WordPress uploads directory (e.g., wp-content/uploads/)', 'ftp-media'); ?></li>
                    <li><?php _e('Ensure your files maintain the same directory structure as they had on the FTP server (e.g., 2025/10/filename.jpg)', 'ftp-media'); ?></li>
                    <li><?php _e('Once your files are locally available, click the button below to update all media URLs and metadata from FTP to local host', 'ftp-media'); ?></li>
                </ol>
                <p><strong style="color: #d63638;"><?php _e('Warning:', 'ftp-media'); ?></strong> <?php _e('This action cannot be undone. Make sure you have backed up your database before proceeding.', 'ftp-media'); ?></p>
                <button type="button" id="migrate-ftp-to-local" class="button button-secondary" style="margin-top: 10px;">
                    <?php _e('Migrate FTP URLs to Local', 'ftp-media'); ?>
                </button>
                <div id="migration-result" style="margin-top: 10px;"></div>
            </div>
        </div>
        <?php
    }
    
    public function main_section_callback() {
        echo '<p>' . __('Configure your FTP server settings below. All media uploads will be redirected to your FTP server.', 'ftp-media') . '</p>';
    }
    
    public function enabled_callback() {
        $settings = get_option('ftp_media_settings');
        $enabled = isset($settings['enabled']) ? $settings['enabled'] : false;
        ?>
        <input type="checkbox" name="ftp_media_settings[enabled]" value="1" <?php checked($enabled, 1); ?> />
        <label for="ftp_media_settings[enabled]"><?php _e('Enable FTP Media uploads', 'ftp-media'); ?></label>
        <?php
    }
    
    public function ftp_host_callback() {
        $settings = get_option('ftp_media_settings');
        $host = isset($settings['ftp_host']) ? $settings['ftp_host'] : '';
        ?>
        <input type="text" name="ftp_media_settings[ftp_host]" value="<?php echo esc_attr($host); ?>" class="regular-text" />
        <p class="description"><?php _e('Enter your FTP server hostname or IP address', 'ftp-media'); ?></p>
        <?php
    }
    
    public function ftp_username_callback() {
        $settings = get_option('ftp_media_settings');
        $username = isset($settings['ftp_username']) ? $settings['ftp_username'] : '';
        ?>
        <input type="text" name="ftp_media_settings[ftp_username]" value="<?php echo esc_attr($username); ?>" class="regular-text" />
        <p class="description"><?php _e('Enter your FTP username', 'ftp-media'); ?></p>
        <?php
    }
    
    public function ftp_password_callback() {
        $settings = get_option('ftp_media_settings');
        $password = isset($settings['ftp_password']) ? $settings['ftp_password'] : '';
        ?>
        <input type="password" name="ftp_media_settings[ftp_password]" value="<?php echo esc_attr($password); ?>" class="regular-text" />
        <p class="description"><?php _e('Enter your FTP password', 'ftp-media'); ?></p>
        <?php
    }
    
    public function ftp_port_callback() {
        $settings = get_option('ftp_media_settings');
        $port = isset($settings['ftp_port']) ? $settings['ftp_port'] : 21;
        ?>
        <input type="number" name="ftp_media_settings[ftp_port]" value="<?php echo esc_attr($port); ?>" class="small-text" min="1" max="65535" />
        <p class="description"><?php _e('FTP port (default: 21, SFTP: 22)', 'ftp-media'); ?></p>
        <?php
    }
    
    public function ftp_directory_callback() {
        $settings = get_option('ftp_media_settings');
        $directory = isset($settings['ftp_directory']) ? $settings['ftp_directory'] : '/';
        ?>
        <input type="text" name="ftp_media_settings[ftp_directory]" value="<?php echo esc_attr($directory); ?>" class="regular-text" />
        <p class="description"><?php _e('Directory on FTP server where files will be uploaded (e.g., /public_html/media/)', 'ftp-media'); ?></p>
        <?php
    }
    
    public function ftp_url_callback() {
        $settings = get_option('ftp_media_settings');
        $url = isset($settings['ftp_url']) ? $settings['ftp_url'] : '';
        ?>
        <input type="url" name="ftp_media_settings[ftp_url]" value="<?php echo esc_attr($url); ?>" class="regular-text" />
        <p class="description"><?php _e('Public URL to access files on FTP server (e.g., https://yourdomain.com/media/)', 'ftp-media'); ?></p>
        <?php
    }
    
    public function ftp_passive_callback() {
        $settings = get_option('ftp_media_settings');
        $passive = isset($settings['ftp_passive']) ? $settings['ftp_passive'] : false;
        ?>
        <input type="checkbox" name="ftp_media_settings[ftp_passive]" value="1" <?php checked($passive, 1); ?> />
        <label for="ftp_media_settings[ftp_passive]"><?php _e('Use passive mode', 'ftp-media'); ?></label>
        <p class="description"><?php _e('Recommended for most FTP servers', 'ftp-media'); ?></p>
        <?php
    }
    
    
    public function validate_settings($input) {
        $output = array();
        
        $output['enabled'] = isset($input['enabled']) ? (bool) $input['enabled'] : false;
        $output['ftp_host'] = sanitize_text_field($input['ftp_host']);
        $output['ftp_username'] = sanitize_text_field($input['ftp_username']);
        $output['ftp_password'] = sanitize_text_field($input['ftp_password']);
        $output['ftp_port'] = absint($input['ftp_port']) ?: 21;
        $output['ftp_directory'] = sanitize_text_field($input['ftp_directory']);
        $output['ftp_url'] = esc_url_raw($input['ftp_url']);
        $output['ftp_passive'] = isset($input['ftp_passive']) ? (bool) $input['ftp_passive'] : false;
        
        return $output;
    }
}
