<?php
/**
 * FTP Uploader Class
 */

if (!defined('ABSPATH')) {
    exit;
}

class FTP_Uploader {
    
    private $settings;
    private $connection;
    private $logger;
    
    public function __construct() {
        $this->settings = get_option('ftp_media_settings');
        $this->logger = new FTP_Logger();
        
        // Force non-SSL connection
        $this->settings['ftp_ssl'] = false;
    }
    
    /**
     * Upload file to FTP server
     */
    public function upload_to_ftp($local_file, $original_url) {
        if (!$this->settings['enabled']) {
            $this->logger->info('FTP Media is disabled, skipping upload');
            return array('success' => false, 'message' => 'FTP Media is disabled');
        }

        // Check if FTP settings are properly configured
        if (empty($this->settings['ftp_host']) || empty($this->settings['ftp_username'])) {
            $this->logger->info('FTP settings not configured, skipping upload');
            return array('success' => false, 'message' => 'FTP settings not configured');
        }

        $this->logger->info("Starting FTP upload for file: {$local_file}");
        $this->logger->info("FTP Settings - Host: {$this->settings['ftp_host']}, Port: {$this->settings['ftp_port']}, SSL: " . ($this->settings['ftp_ssl'] ? 'Yes' : 'No'));
        $this->logger->info("Full settings: " . print_r($this->settings, true));
        
        // Check if file exists before proceeding
        if (!file_exists($local_file)) {
            $this->logger->error("Local file does not exist: {$local_file}");
            return array('success' => false, 'message' => 'Local file does not exist: ' . $local_file);
        }
        
        $this->logger->info("Local file exists, size: " . filesize($local_file) . " bytes");

        // Wait for file to exist (retry mechanism)
        $max_retries = 5;
        $retry_delay = 1; // seconds
        
        for ($i = 0; $i < $max_retries; $i++) {
            if (file_exists($local_file)) {
                $this->logger->info("File found after {$i} retries: {$local_file}");
                break;
            }
            
            if ($i < $max_retries - 1) {
                $this->logger->info("File not found, retrying in {$retry_delay} seconds... (attempt " . ($i + 1) . "/{$max_retries})");
                sleep($retry_delay);
            }
        }

        if (!file_exists($local_file)) {
            $this->logger->error("File does not exist after {$max_retries} retries: {$local_file}");
            return array('success' => false, 'message' => 'File does not exist: ' . $local_file);
        }

        try {
            $this->logger->info('Starting try block in upload_to_ftp');
            
            // Connect to FTP server
            $this->logger->info('About to call connect() method');
            $this->connect();
            $this->logger->info('Connect method completed');

            if (!$this->connection) {
                $this->logger->error('Failed to connect to FTP server');
                return array('success' => false, 'message' => 'Failed to connect to FTP server');
            }

            $this->logger->info('Successfully connected to FTP server');
            
            // Test basic FTP operations
            $pwd = @ftp_pwd($this->connection);
            $this->logger->info("Current FTP directory: " . ($pwd ? $pwd : 'Unknown'));
            
            // Test directory listing
            $listing = @ftp_nlist($this->connection, '.');
            $this->logger->info("Directory listing test: " . (is_array($listing) ? 'Success' : 'Failed'));

            $this->logger->info('About to generate WordPress directory structure');
            
            // Generate WordPress-style directory structure (year/month)
            $upload_dir = wp_upload_dir();
            $this->logger->info("Upload dir basedir: " . $upload_dir['basedir']);
            $this->logger->info("Local file path: " . $local_file);
            
            $relative_path = str_replace($upload_dir['basedir'], '', $local_file);
            $this->logger->info("Relative path after replace: " . $relative_path);
            
            $relative_path = ltrim($relative_path, '/');
            $this->logger->info("Final relative path: " . $relative_path);

            // Create the full remote path
            $remote_path = $this->get_remote_path($relative_path);

            $this->logger->info("Uploading file to remote path: {$remote_path}");
            $this->logger->info("About to test FTP upload capabilities...");
            
            // Test upload with a small file first
            $test_file = 'test_' . time() . '.txt';
            $test_content = 'FTP test file';
            $temp_test = tempnam(sys_get_temp_dir(), 'ftp_test');
            file_put_contents($temp_test, $test_content);
            
            $this->logger->info("Testing upload with small file: {$test_file}");
            $this->logger->info("Test file created at: {$temp_test}");
            $this->logger->info("Test file size: " . filesize($temp_test) . " bytes");
            
            $test_upload = @ftp_put($this->connection, $test_file, $temp_test, FTP_ASCII);
            unlink($temp_test);
            
            $this->logger->info("Test upload result: " . ($test_upload ? 'SUCCESS' : 'FAILED'));
            
            if ($test_upload) {
                $this->logger->info("Connection status before ftp_delete: " . (is_resource($this->connection) ? 'Valid' : 'Invalid'));
                @ftp_delete($this->connection, $test_file);
                $this->logger->info("Connection status after ftp_delete: " . (is_resource($this->connection) ? 'Valid' : 'Invalid'));
                $this->logger->info("Test upload successful, proceeding with main file");
            } else {
                $this->logger->error("Test upload failed - FTP server may not allow uploads");
                $error = error_get_last();
                $error_msg = $error ? $error['message'] : 'Unknown error';
                $this->logger->error("Test upload error: {$error_msg}");
                throw new Exception('FTP server does not allow file uploads: ' . $error_msg);
            }

            // Upload file
            $this->logger->info("About to call upload_file method...");
            $this->logger->info("Local file: {$local_file}");
            $this->logger->info("Remote path: {$remote_path}");
            
            $actual_remote_path = $this->upload_file($local_file, $remote_path);
            
            $this->logger->info("Upload file method returned: " . ($actual_remote_path ? 'SUCCESS' : 'FAILED'));
            $this->logger->info("Actual remote path used: " . var_export($actual_remote_path, true));

            if ($actual_remote_path) {
                // Strip the directory prefix to get the relative path for URL generation
                $directory = trim($this->settings['ftp_directory'], '/');
                $url_relative_path = $actual_remote_path;
                
                if ($directory && $directory !== '/') {
                    $url_relative_path = str_replace($directory . '/', '', $actual_remote_path);
                }
                
                // Generate FTP URL using the actual path
                $ftp_url = $this->get_ftp_url($url_relative_path);

                $this->logger->info("File uploaded successfully. FTP URL: {$ftp_url}");

                // Disconnect
                $this->disconnect();

                return array(
                    'success' => true,
                    'ftp_url' => $ftp_url,
                    'remote_path' => $actual_remote_path
                );
            } else {
                $this->logger->error('Failed to upload file to FTP server');
                $this->disconnect();
                return array('success' => false, 'message' => 'Failed to upload file to FTP server');
            }

        } catch (Exception $e) {
            $this->logger->error("FTP upload failed: " . $e->getMessage());
            $this->disconnect();
            return array('success' => false, 'message' => $e->getMessage());
        }
    }
    
    /**
     * Connect to FTP server
     */
    private function connect() {
        $host = $this->settings['ftp_host'];
        $username = $this->settings['ftp_username'];
        $password = $this->settings['ftp_password'];
        $port = $this->settings['ftp_port'];
        $ssl = $this->settings['ftp_ssl'];
        
        // Validate required fields
        if (empty($host)) {
            throw new Exception('FTP Host is required');
        }
        if (empty($username)) {
            throw new Exception('FTP Username is required');
        }
        if (empty($password)) {
            throw new Exception('FTP Password is required');
        }
        
        $this->logger->info("Attempting to connect to FTP server: {$host}:{$port} (SSL: " . ($ssl ? 'Yes' : 'No') . ")");
        
        // Test connection
        if ($ssl) {
            if (!function_exists('ftp_ssl_connect')) {
                throw new Exception('FTPS (SSL) is not supported on this server. Please disable SSL or contact your hosting provider.');
            }
            $this->connection = @ftp_ssl_connect($host, $port, 30);
        } else {
            if (!function_exists('ftp_connect')) {
                throw new Exception('FTP functions are not available on this server. Please contact your hosting provider.');
            }
            $this->connection = @ftp_connect($host, $port, 30);
        }
        
        if (!$this->connection) {
            $error = error_get_last();
            $error_msg = $error ? $error['message'] : 'Unknown error';
            
            // Provide specific error messages based on common issues
            if (strpos($error_msg, 'Connection refused') !== false) {
                throw new Exception("Connection refused to '{$host}:{$port}'. The server is not accepting connections. Please check: 1) Host address is correct, 2) Port number is correct, 3) Server is running, 4) Firewall allows connections.");
            } elseif (strpos($error_msg, 'Connection timed out') !== false) {
                throw new Exception("Connection timed out to '{$host}:{$port}'. The server is not responding. Please check: 1) Host address is correct, 2) Network connectivity, 3) Server is running, 4) Firewall settings.");
            } elseif (strpos($error_msg, 'Name or service not known') !== false) {
                throw new Exception("Cannot resolve hostname '{$host}'. Please check: 1) Host address is spelled correctly, 2) DNS is working, 3) Try using IP address instead of hostname.");
            } else {
                throw new Exception("Could not connect to FTP server '{$host}:{$port}'. Error: {$error_msg}. Please check your host and port settings.");
            }
        }
        
        $this->logger->info("FTP connection established, attempting login");
        
        $login = @ftp_login($this->connection, $username, $password);
        if (!$login) {
            $error = error_get_last();
            $error_msg = $error ? $error['message'] : 'Unknown error';
            ftp_close($this->connection);
            $this->connection = null;
            
            // Provide specific login error messages
            if (strpos($error_msg, 'Login incorrect') !== false || strpos($error_msg, '530') !== false) {
                throw new Exception("Login failed for user '{$username}'. Please check: 1) Username is correct, 2) Password is correct, 3) User account is active, 4) User has FTP access permissions.");
            } elseif (strpos($error_msg, '421') !== false) {
                throw new Exception("FTP server is too busy or has reached connection limit. Please try again in a few minutes.");
            } else {
                throw new Exception("Could not login to FTP server. Please check your username and password. Error: {$error_msg}");
            }
        }
        
        $this->logger->info("FTP login successful");
        
        // Set passive mode if enabled
        if ($this->settings['ftp_passive']) {
            if (!@ftp_pasv($this->connection, true)) {
                $this->logger->warning("Could not set passive mode, continuing with active mode");
            } else {
                $this->logger->info("Passive mode enabled");
            }
        }
        
        // Set timeout for data operations (upload/download) to 300 seconds (5 minutes)
        @ftp_set_option($this->connection, FTP_TIMEOUT_SEC, 300);
        $this->logger->info("FTP timeout set to 300 seconds for large file transfers");
        
        // Change to specified directory
        $directory = $this->settings['ftp_directory'];
        if ($directory && $directory !== '/') {
            $this->logger->info("Attempting to change to directory: {$directory}");
            if (!@ftp_chdir($this->connection, $directory)) {
                $this->logger->info("Directory does not exist, attempting to create: {$directory}");
                try {
                    $this->create_directory($directory);
                    if (!@ftp_chdir($this->connection, $directory)) {
                        throw new Exception("Could not access directory '{$directory}'. Please check: 1) Directory path is correct, 2) Directory exists, 3) You have read/write permissions, 4) Directory is not full.");
                    }
                } catch (Exception $e) {
                    if (strpos($e->getMessage(), 'Permission denied') !== false) {
                        throw new Exception("Permission denied for directory '{$directory}'. Please check: 1) You have write permissions, 2) Directory ownership is correct, 3) FTP user has access to parent directories.");
                    } else {
                        throw new Exception("Could not create or access directory '{$directory}'. Error: " . $e->getMessage());
                    }
                }
            }
            $this->logger->info("Successfully changed to directory: {$directory}");
        }
    }
    
    /**
     * Disconnect from FTP server
     */
    private function disconnect() {
        if ($this->connection) {
            ftp_close($this->connection);
            $this->connection = null;
        }
    }
    
    /**
     * Upload file to FTP server
     */
    private function upload_file($local_file, $remote_path) {
        if (!file_exists($local_file)) {
            throw new Exception('Local file does not exist: ' . $local_file);
        }

        $this->logger->info("Attempting to upload file: {$local_file} to {$remote_path}");
        $this->logger->info("File size: " . filesize($local_file) . " bytes");
        $this->logger->info("Remote path: {$remote_path}");
        $this->logger->info("Local file: {$local_file}");
        
        // Ensure the remote directory exists
        $remote_dir = dirname($remote_path);
        if ($remote_dir !== '.') {
            $this->logger->info("Ensuring remote directory exists: {$remote_dir}");
            try {
                $this->create_directory($remote_dir);
                $this->logger->info("Directory creation completed successfully");
                
                // Return to root directory after creating directories
                $this->logger->info("About to return to root directory, connection status: " . (is_resource($this->connection) ? 'Valid' : 'Invalid'));
                
                $chdir_result = @ftp_chdir($this->connection, '/');
                $this->logger->info("chdir result: " . ($chdir_result ? 'Success' : 'Failed'));
                $this->logger->info("Connection status after chdir: " . (is_resource($this->connection) ? 'Valid' : 'Invalid'));
                
                $pwd = @ftp_pwd($this->connection);
                $this->logger->info("Returned to root directory: {$pwd}");
                $this->logger->info("Connection status after pwd: " . (is_resource($this->connection) ? 'Valid' : 'Invalid'));
            } catch (Exception $e) {
                $this->logger->error("Directory creation failed: " . $e->getMessage());
                throw new Exception('Failed to create directory: ' . $e->getMessage());
            }
        }
        
        // Check if file exists and get unique filename if needed
        $remote_path = $this->get_unique_remote_path($remote_path);
        
        // Test connection before upload
        $this->logger->info("Testing connection before upload, connection status: " . (is_resource($this->connection) ? 'Valid' : 'Invalid'));
        
        $pwd = @ftp_pwd($this->connection);
        if ($pwd === false) {
            $this->logger->error("FTP connection lost before upload");
            throw new Exception('FTP connection lost before upload');
        }
        $this->logger->info("FTP connection verified, current directory: {$pwd}");
        $this->logger->info("Connection status after verification: " . (is_resource($this->connection) ? 'Valid' : 'Invalid'));
        
        // Try multiple upload methods
        $this->logger->info("Attempting FTP upload with BINARY mode...");
        $this->logger->info("Full remote path for upload: {$remote_path}");
        
        // Note: On some servers, is_resource() may return false even though the connection works
        // So we just try the upload without checking is_resource()
        
        $upload_result = @ftp_put($this->connection, $remote_path, $local_file, FTP_BINARY);
        
        if (!$upload_result) {
            $this->logger->info("BINARY mode failed, trying ASCII mode...");
            $upload_result = @ftp_put($this->connection, $remote_path, $local_file, FTP_ASCII);
        }
        
        if (!$upload_result) {
            $this->logger->info("ASCII mode failed, trying without mode specification...");
            $upload_result = @ftp_put($this->connection, $remote_path, $local_file);
        }

        if (!$upload_result) {
            $error = error_get_last();
            $error_msg = $error ? $error['message'] : 'Unknown error';
            $this->logger->error("FTP upload failed. Error: {$error_msg}");
            $this->logger->error("Connection status: " . (is_resource($this->connection) ? 'Active' : 'Closed'));
            
            // Try to get more details about the failure
            $pwd = @ftp_pwd($this->connection);
            $this->logger->error("Current FTP directory: " . ($pwd ? $pwd : 'Unknown'));
            
            // Test if we can create a simple test file
            $test_file = 'test_' . time() . '.txt';
            $test_content = 'test';
            $temp_test = tempnam(sys_get_temp_dir(), 'ftp_test');
            file_put_contents($temp_test, $test_content);
            
            $this->logger->info("Testing with simple text file: {$test_file}");
            $test_upload = @ftp_put($this->connection, $test_file, $temp_test, FTP_ASCII);
            unlink($temp_test);
            
            if ($test_upload) {
                @ftp_delete($this->connection, $test_file);
                $this->logger->error("Test file upload succeeded, but main file failed. Possible file size or permission issue.");
                throw new Exception('FTP server accepts files but upload failed. Possible file size or permission issue: ' . $error_msg);
            } else {
                $this->logger->error("Even test file upload failed. FTP server may not allow file uploads.");
                throw new Exception('FTP server does not allow file uploads. Check permissions: ' . $error_msg);
            }
        }

        $this->logger->info("File uploaded successfully: {$remote_path}");
        
        // Return the actual remote path used (which may be different if renamed for uniqueness)
        return $remote_path;
    }
    
    /**
     * Get remote path for file
     */
    private function get_remote_path($relative_path) {
        $directory = $this->settings['ftp_directory'];
        $directory = rtrim($directory, '/');

        if ($directory && $directory !== '/') {
            return $directory . '/' . $relative_path;
        }

        return $relative_path;
    }

    /**
     * Get FTP URL for file
     */
    private function get_ftp_url($relative_path) {
        $base_url = rtrim($this->settings['ftp_url'], '/');
        $directory = trim($this->settings['ftp_directory'], '/');

        if ($directory && $directory !== '/') {
            return $base_url . '/' . $directory . '/' . $relative_path;
        }

        return $base_url . '/' . $relative_path;
    }
    
    /**
     * Create directory on FTP server
     */
    private function create_directory($directory) {
        $this->logger->info("Creating directory: {$directory}");
        $this->logger->info("Connection status at start of create_directory: " . (is_resource($this->connection) ? 'Valid' : 'Invalid'));
        
        $parts = explode('/', trim($directory, '/'));
        $current_path = '';
        
        foreach ($parts as $part) {
            if (empty($part)) continue;
            
            $current_path .= '/' . $part;
            $this->logger->info("Checking directory: {$current_path}");
            $this->logger->info("Connection status before ftp_nlist: " . (is_resource($this->connection) ? 'Valid' : 'Invalid'));
            
            // Use ftp_nlist to check if directory exists (doesn't change connection state)
            $listing = @ftp_nlist($this->connection, $current_path);
            
            $this->logger->info("Connection status after ftp_nlist: " . (is_resource($this->connection) ? 'Valid' : 'Invalid'));
            $this->logger->info("ftp_nlist result: " . ($listing === false ? 'false' : 'array with ' . count($listing) . ' items'));
            
            if ($listing === false) {
                // Directory doesn't exist, create it
                $this->logger->info("Directory does not exist, creating: {$current_path}");
                if (!@ftp_mkdir($this->connection, $current_path)) {
                    $error = error_get_last();
                    $error_msg = $error ? $error['message'] : 'Unknown error';
                    $this->logger->error("Failed to create directory {$current_path}: {$error_msg}");
                    throw new Exception('Could not create directory: ' . $current_path . ' - ' . $error_msg);
                } else {
                    $this->logger->info("Successfully created directory: {$current_path}");
                }
            } else {
                $this->logger->info("Directory already exists: {$current_path}");
            }
            
            $this->logger->info("Connection status at end of loop iteration: " . (is_resource($this->connection) ? 'Valid' : 'Invalid'));
        }
        
        $this->logger->info("Directory creation completed for: {$directory}");
        $this->logger->info("Connection status at end of create_directory: " . (is_resource($this->connection) ? 'Valid' : 'Invalid'));
    }
    
    /**
     * Test FTP connection and directory creation
     */
    public function test_ftp_operations() {
        try {
            $this->connect();
            
            if (!$this->connection) {
                return array('success' => false, 'message' => 'Failed to connect to FTP server');
            }
            
            $this->logger->info('Testing FTP operations...');
            
            // Test basic operations
            $pwd = @ftp_pwd($this->connection);
            $this->logger->info("Current directory: " . ($pwd ? $pwd : 'Unknown'));
            
            // Test directory creation
            $test_dir = 'test_' . time();
            $this->logger->info("Testing directory creation: {$test_dir}");
            
            if (@ftp_mkdir($this->connection, $test_dir)) {
                $this->logger->info("Directory created successfully: {$test_dir}");
                
                // Test directory change
                if (@ftp_chdir($this->connection, $test_dir)) {
                    $this->logger->info("Successfully changed to directory: {$test_dir}");
                    @ftp_chdir($this->connection, '..'); // Go back
                }
                
                // Clean up - remove test directory
                @ftp_rmdir($this->connection, $test_dir);
                $this->logger->info("Test directory removed: {$test_dir}");
            } else {
                $error = error_get_last();
                $error_msg = $error ? $error['message'] : 'Unknown error';
                $this->logger->error("Failed to create test directory: {$error_msg}");
                return array('success' => false, 'message' => 'Failed to create directory: ' . $error_msg);
            }
            
            $this->disconnect();
            return array('success' => true, 'message' => 'FTP operations test successful');
            
        } catch (Exception $e) {
            $this->logger->error("FTP operations test failed: " . $e->getMessage());
            $this->disconnect();
            return array('success' => false, 'message' => $e->getMessage());
        }
    }
    
    /**
     * Test FTP connection
     */
    public function test_connection() {
        $this->logger->info('Testing FTP connection');
        
        try {
            // Validate settings first
            if (empty($this->settings['ftp_host'])) {
                throw new Exception('FTP Host is required');
            }
            if (empty($this->settings['ftp_username'])) {
                throw new Exception('FTP Username is required');
            }
            if (empty($this->settings['ftp_password'])) {
                throw new Exception('FTP Password is required');
            }
            
            $this->connect();
            
            if ($this->connection) {
                // Test if we can list the current directory
                $files = @ftp_nlist($this->connection, '.');
                if ($files === false) {
                    throw new Exception('Connected but cannot list directory contents. Please check directory permissions.');
                }
                
                // Test if we can create a test file (optional)
                $test_file = 'ftp_media_test_' . time() . '.txt';
                $test_content = 'FTP Media Plugin Test File - ' . date('Y-m-d H:i:s');
                
                if (@ftp_put($this->connection, $test_file, 'data://text/plain,' . $test_content, FTP_ASCII)) {
                    // Clean up test file
                    @ftp_delete($this->connection, $test_file);
                    $this->logger->info('FTP connection test successful - can read and write');
                } else {
                    $this->logger->warning('FTP connection successful but cannot write files. Check write permissions.');
                }
                
                $this->logger->info('FTP connection test successful');
                $this->disconnect();
                return array('success' => true, 'message' => 'FTP connection successful! Server is accessible and ready for file uploads.');
            } else {
                $this->logger->error('FTP connection test failed');
                return array('success' => false, 'message' => 'Failed to connect to FTP server');
            }
        } catch (Exception $e) {
            $this->logger->error('FTP connection test failed: ' . $e->getMessage());
            return array('success' => false, 'message' => $e->getMessage());
        }
    }
    
    /**
     * Check if file exists on FTP server and generate unique filename if needed
     */
    private function get_unique_remote_path($remote_path) {
        // Check if file exists
        $file_size = @ftp_size($this->connection, $remote_path);
        
        if ($file_size === -1) {
            // File doesn't exist, return original path
            $this->logger->info("File does not exist on FTP, using original name: {$remote_path}");
            return $remote_path;
        }
        
        // File exists, generate unique name
        $this->logger->info("File already exists on FTP: {$remote_path}, generating unique name");
        
        $path_info = pathinfo($remote_path);
        $dirname = $path_info['dirname'];
        $filename = $path_info['filename'];
        $extension = isset($path_info['extension']) ? '.' . $path_info['extension'] : '';
        
        // Try incrementing numbers like WordPress does
        $counter = 1;
        do {
            $new_filename = $filename . '-' . $counter . $extension;
            $new_path = ($dirname !== '.' ? $dirname . '/' : '') . $new_filename;
            
            $file_size = @ftp_size($this->connection, $new_path);
            $counter++;
            
            // Prevent infinite loop
            if ($counter > 100) {
                // Use timestamp as fallback
                $new_filename = $filename . '-' . time() . $extension;
                $new_path = ($dirname !== '.' ? $dirname . '/' : '') . $new_filename;
                break;
            }
        } while ($file_size !== -1);
        
        $this->logger->info("Generated unique filename: {$new_path}");
        return $new_path;
    }
    
    /**
     * Delete file from FTP server
     */
    public function delete_from_ftp($ftp_url) {
        if (!$this->settings['enabled']) {
            $this->logger->info('FTP Media is disabled, skipping delete');
            return array('success' => false, 'message' => 'FTP Media is disabled');
        }

        if (empty($this->settings['ftp_host']) || empty($this->settings['ftp_username'])) {
            $this->logger->info('FTP settings not configured, skipping delete');
            return array('success' => false, 'message' => 'FTP settings not configured');
        }

        $this->logger->info("Starting FTP delete for URL: {$ftp_url}");

        try {
            // Connect to FTP server
            $this->connect();

            if (!$this->connection) {
                $this->logger->error('Failed to connect to FTP server');
                return array('success' => false, 'message' => 'Failed to connect to FTP server');
            }

            // Extract the file path from the FTP URL
            $base_url = rtrim($this->settings['ftp_url'], '/');
            $remote_path = str_replace($base_url, '', $ftp_url);
            $remote_path = ltrim($remote_path, '/');
            
            // Add directory prefix if configured
            $directory = trim($this->settings['ftp_directory'], '/');
            if ($directory && $directory !== '/') {
                $remote_path = $directory . '/' . $remote_path;
            }

            $this->logger->info("Deleting file from remote path: {$remote_path}");

            // Delete the file
            $delete_result = @ftp_delete($this->connection, $remote_path);

            if ($delete_result) {
                $this->logger->info("Successfully deleted file: {$remote_path}");
                $this->disconnect();
                return array('success' => true, 'message' => 'File deleted successfully');
            } else {
                $this->logger->error("Failed to delete file: {$remote_path}");
                $this->disconnect();
                return array('success' => false, 'message' => 'Failed to delete file from FTP server');
            }

        } catch (Exception $e) {
            $this->logger->error("FTP delete failed: " . $e->getMessage());
            $this->disconnect();
            return array('success' => false, 'message' => $e->getMessage());
        }
    }
}
